import { Scope } from '@nestjs/common';

import { Not } from 'typeorm';
import { type ObjectRecordDeleteEvent } from 'twenty-shared/database-events';

import { Process } from 'src/engine/core-modules/message-queue/decorators/process.decorator';
import { Processor } from 'src/engine/core-modules/message-queue/decorators/processor.decorator';
import { MessageQueue } from 'src/engine/core-modules/message-queue/message-queue.constants';
import { GlobalWorkspaceOrmManager } from 'src/engine/twenty-orm/global-workspace-datasource/global-workspace-orm.manager';
import { buildSystemAuthContext } from 'src/engine/twenty-orm/utils/build-system-auth-context.util';
import { type WorkspaceEventBatch } from 'src/engine/workspace-event-emitter/types/workspace-event-batch.type';
import { type BlocklistWorkspaceEntity } from 'src/modules/blocklist/standard-objects/blocklist.workspace-entity';
import { MessageChannelSyncStatusService } from 'src/modules/messaging/common/services/message-channel-sync-status.service';
import {
  MessageChannelSyncStage,
  type MessageChannelWorkspaceEntity,
} from 'src/modules/messaging/common/standard-objects/message-channel.workspace-entity';

export type BlocklistReimportMessagesJobData = WorkspaceEventBatch<
  ObjectRecordDeleteEvent<BlocklistWorkspaceEntity>
>;

@Processor({
  queueName: MessageQueue.messagingQueue,
  scope: Scope.REQUEST,
})
export class BlocklistReimportMessagesJob {
  constructor(
    private readonly globalWorkspaceOrmManager: GlobalWorkspaceOrmManager,
    private readonly messagingChannelSyncStatusService: MessageChannelSyncStatusService,
  ) {}

  @Process(BlocklistReimportMessagesJob.name)
  async handle(data: BlocklistReimportMessagesJobData): Promise<void> {
    const workspaceId = data.workspaceId;

    const authContext = buildSystemAuthContext(workspaceId);

    await this.globalWorkspaceOrmManager.executeInWorkspaceContext(async () => {
      const messageChannelRepository =
        await this.globalWorkspaceOrmManager.getRepository<MessageChannelWorkspaceEntity>(
          workspaceId,
          'messageChannel',
        );

      for (const eventPayload of data.events) {
        const workspaceMemberId =
          eventPayload.properties.before.workspaceMemberId;

        const messageChannels = await messageChannelRepository.find({
          select: ['id'],
          where: {
            connectedAccount: {
              accountOwnerId: workspaceMemberId,
            },
            syncStage: Not(MessageChannelSyncStage.MESSAGE_LIST_FETCH_PENDING),
          },
        });

        await this.messagingChannelSyncStatusService.resetAndMarkAsMessagesListFetchPending(
          messageChannels.map((messageChannel) => messageChannel.id),
          workspaceId,
        );
      }
    }, authContext);
  }
}
