import { Scope } from '@nestjs/common';

import { Not } from 'typeorm';
import { type ObjectRecordDeleteEvent } from 'twenty-shared/database-events';

import { Process } from 'src/engine/core-modules/message-queue/decorators/process.decorator';
import { Processor } from 'src/engine/core-modules/message-queue/decorators/processor.decorator';
import { MessageQueue } from 'src/engine/core-modules/message-queue/message-queue.constants';
import { GlobalWorkspaceOrmManager } from 'src/engine/twenty-orm/global-workspace-datasource/global-workspace-orm.manager';
import { buildSystemAuthContext } from 'src/engine/twenty-orm/utils/build-system-auth-context.util';
import { type WorkspaceEventBatch } from 'src/engine/workspace-event-emitter/types/workspace-event-batch.type';
import { type BlocklistWorkspaceEntity } from 'src/modules/blocklist/standard-objects/blocklist.workspace-entity';
import { CalendarChannelSyncStatusService } from 'src/modules/calendar/common/services/calendar-channel-sync-status.service';
import {
  CalendarChannelSyncStage,
  type CalendarChannelWorkspaceEntity,
} from 'src/modules/calendar/common/standard-objects/calendar-channel.workspace-entity';

export type BlocklistReimportCalendarEventsJobData = WorkspaceEventBatch<
  ObjectRecordDeleteEvent<BlocklistWorkspaceEntity>
>;

@Processor({
  queueName: MessageQueue.calendarQueue,
  scope: Scope.REQUEST,
})
export class BlocklistReimportCalendarEventsJob {
  constructor(
    private readonly globalWorkspaceOrmManager: GlobalWorkspaceOrmManager,
    private readonly calendarChannelSyncStatusService: CalendarChannelSyncStatusService,
  ) {}

  @Process(BlocklistReimportCalendarEventsJob.name)
  async handle(data: BlocklistReimportCalendarEventsJobData): Promise<void> {
    const workspaceId = data.workspaceId;

    const authContext = buildSystemAuthContext(workspaceId);

    await this.globalWorkspaceOrmManager.executeInWorkspaceContext(async () => {
      const calendarChannelRepository =
        await this.globalWorkspaceOrmManager.getRepository<CalendarChannelWorkspaceEntity>(
          workspaceId,
          'calendarChannel',
        );

      for (const eventPayload of data.events) {
        const workspaceMemberId =
          eventPayload.properties.before.workspaceMemberId;

        const calendarChannels = await calendarChannelRepository.find({
          select: ['id'],
          where: {
            connectedAccount: {
              accountOwnerId: workspaceMemberId,
            },
            syncStage: Not(
              CalendarChannelSyncStage.CALENDAR_EVENT_LIST_FETCH_PENDING,
            ),
          },
        });

        await this.calendarChannelSyncStatusService.resetAndMarkAsCalendarEventListFetchPending(
          calendarChannels.map((calendarChannel) => calendarChannel.id),
          workspaceId,
        );
      }
    }, authContext);
  }
}
