import {
  DateDisplayFormat,
  FieldMetadataType,
  RelationOnDeleteAction,
  RelationType,
} from 'twenty-shared/types';

import { type FlatFieldMetadata } from 'src/engine/metadata-modules/flat-field-metadata/types/flat-field-metadata.type';
import { type AllStandardObjectFieldName } from 'src/engine/workspace-manager/twenty-standard-application/types/all-standard-object-field-name.type';
import {
  type CreateStandardFieldArgs,
  createStandardFieldFlatMetadata,
} from 'src/engine/workspace-manager/twenty-standard-application/utils/field-metadata/create-standard-field-flat-metadata.util';
import { createStandardRelationFieldFlatMetadata } from 'src/engine/workspace-manager/twenty-standard-application/utils/field-metadata/create-standard-relation-field-flat-metadata.util';
import { getTsVectorColumnExpressionFromFields } from 'src/engine/workspace-manager/utils/get-ts-vector-column-expression.util';
import { SEARCH_FIELDS_FOR_WORKFLOW_RUNS } from 'src/modules/workflow/common/standard-objects/workflow-run.workspace-entity';

export const buildWorkflowRunStandardFlatFieldMetadatas = ({
  now,
  objectName,
  workspaceId,
  standardObjectMetadataRelatedEntityIds,
  dependencyFlatEntityMaps,
  twentyStandardApplicationId,
}: Omit<
  CreateStandardFieldArgs<'workflowRun', FieldMetadataType>,
  'context'
>): Record<AllStandardObjectFieldName<'workflowRun'>, FlatFieldMetadata> => ({
  id: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'id',
      type: FieldMetadataType.UUID,
      label: 'Id',
      description: 'Id',
      icon: 'Icon123',
      isSystem: true,
      isNullable: false,
      isUIReadOnly: true,
      defaultValue: 'uuid',
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  createdAt: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'createdAt',
      type: FieldMetadataType.DATE_TIME,
      label: 'Creation date',
      description: 'Creation date',
      icon: 'IconCalendar',
      isSystem: true,
      isNullable: false,
      isUIReadOnly: true,
      defaultValue: 'now',
      settings: { displayFormat: DateDisplayFormat.RELATIVE },
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  updatedAt: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'updatedAt',
      type: FieldMetadataType.DATE_TIME,
      label: 'Last update',
      description: 'Last time the record was changed',
      icon: 'IconCalendarClock',
      isSystem: true,
      isNullable: false,
      isUIReadOnly: true,
      defaultValue: 'now',
      settings: { displayFormat: DateDisplayFormat.RELATIVE },
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  deletedAt: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'deletedAt',
      type: FieldMetadataType.DATE_TIME,
      label: 'Deleted at',
      description: 'Date when the record was deleted',
      icon: 'IconCalendarMinus',
      isSystem: true,
      isNullable: true,
      isUIReadOnly: true,
      settings: { displayFormat: DateDisplayFormat.RELATIVE },
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  name: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'name',
      type: FieldMetadataType.TEXT,
      label: 'Name',
      description: 'Name of the workflow run',
      icon: 'IconSettingsAutomation',
      isNullable: true,
      isUIReadOnly: true,
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  enqueuedAt: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'enqueuedAt',
      type: FieldMetadataType.DATE_TIME,
      label: 'Workflow run enqueued at',
      description: 'Workflow run enqueued at',
      icon: 'IconHistory',
      isNullable: true,
      isUIReadOnly: true,
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  startedAt: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'startedAt',
      type: FieldMetadataType.DATE_TIME,
      label: 'Workflow run started at',
      description: 'Workflow run started at',
      icon: 'IconHistory',
      isNullable: true,
      isUIReadOnly: true,
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  endedAt: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'endedAt',
      type: FieldMetadataType.DATE_TIME,
      label: 'Workflow run ended at',
      description: 'Workflow run ended at',
      icon: 'IconHistory',
      isNullable: true,
      isUIReadOnly: true,
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  status: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'status',
      type: FieldMetadataType.SELECT,
      label: 'Workflow run status',
      description: 'Workflow run status',
      icon: 'IconStatusChange',
      isNullable: false,
      isUIReadOnly: true,
      defaultValue: "'NOT_STARTED'",
      options: [
        {
          id: '20202020-2ec6-40d8-b9e1-1b1e567bcca9',
          value: 'NOT_STARTED',
          label: 'Not started',
          position: 0,
          color: 'gray',
        },
        {
          id: '20202020-3166-46be-995a-67cb1f4c41d5',
          value: 'RUNNING',
          label: 'Running',
          position: 1,
          color: 'yellow',
        },
        {
          id: '20202020-cde8-4fb6-840a-2fdc4f021b0c',
          value: 'COMPLETED',
          label: 'Completed',
          position: 2,
          color: 'green',
        },
        {
          id: '20202020-fb77-41c7-bf7c-9be97cce805e',
          value: 'FAILED',
          label: 'Failed',
          position: 3,
          color: 'red',
        },
        {
          id: '20202020-c518-4c95-8255-82a05739c88d',
          value: 'ENQUEUED',
          label: 'Enqueued',
          position: 4,
          color: 'blue',
        },
        {
          id: '20202020-e8df-4314-829d-165e296c4eb6',
          value: 'STOPPING',
          label: 'Stopping',
          position: 5,
          color: 'orange',
        },
        {
          id: '20202020-729b-44f9-a9c7-0bf401a0b51c',
          value: 'STOPPED',
          label: 'Stopped',
          position: 6,
          color: 'gray',
        },
      ],
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  createdBy: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'createdBy',
      type: FieldMetadataType.ACTOR,
      label: 'Executed by',
      description: 'The executor of the workflow',
      icon: 'IconCreativeCommonsSa',
      isSystem: true,
      isNullable: false,
      isUIReadOnly: true,
      defaultValue: {
        source: "'MANUAL'",
        name: "'System'",
        workspaceMemberId: null,
      },
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  updatedBy: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'updatedBy',
      type: FieldMetadataType.ACTOR,
      label: 'Updated by',
      description: 'The workspace member who last updated the record',
      icon: 'IconUserCircle',
      isSystem: true,
      isUIReadOnly: true,
      isNullable: false,
      defaultValue: {
        source: "'MANUAL'",
        name: "'System'",
        workspaceMemberId: null,
      },
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  state: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'state',
      type: FieldMetadataType.RAW_JSON,
      label: 'State',
      description: 'State of the workflow run',
      icon: 'IconHierarchy2',
      isNullable: false,
      isUIReadOnly: true,
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  context: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'context',
      type: FieldMetadataType.RAW_JSON,
      label: 'Context',
      description: 'Context of the workflow run',
      icon: 'IconHierarchy2',
      isNullable: true,
      isUIReadOnly: true,
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  output: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'output',
      type: FieldMetadataType.RAW_JSON,
      label: 'Output',
      description: 'Output of the workflow run',
      icon: 'IconHierarchy2',
      isNullable: true,
      isUIReadOnly: true,
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  position: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'position',
      type: FieldMetadataType.POSITION,
      label: 'Position',
      description: 'Workflow run position',
      icon: 'IconHierarchy2',
      isSystem: true,
      isNullable: false,
      isUIReadOnly: true,
      defaultValue: 0,
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  searchVector: createStandardFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      fieldName: 'searchVector',
      type: FieldMetadataType.TS_VECTOR,
      label: 'Search vector',
      description: 'Field used for full-text search',
      icon: 'IconUser',
      isSystem: true,
      isNullable: true,
      isUIReadOnly: true,
      settings: {
        generatedType: 'STORED',
        asExpression: getTsVectorColumnExpressionFromFields(
          SEARCH_FIELDS_FOR_WORKFLOW_RUNS,
        ),
      },
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  workflowVersion: createStandardRelationFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      type: FieldMetadataType.RELATION,
      morphId: null,
      fieldName: 'workflowVersion',
      label: 'Workflow version',
      description: 'Workflow version linked to the run.',
      icon: 'IconVersions',
      isNullable: false,
      isUIReadOnly: true,
      targetObjectName: 'workflowVersion',
      targetFieldName: 'runs',
      settings: {
        relationType: RelationType.MANY_TO_ONE,
        onDelete: RelationOnDeleteAction.SET_NULL,
        joinColumnName: 'workflowVersionId',
      },
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  workflow: createStandardRelationFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      type: FieldMetadataType.RELATION,
      morphId: null,
      fieldName: 'workflow',
      label: 'Workflow',
      description: 'Workflow linked to the run.',
      icon: 'IconSettingsAutomation',
      isNullable: false,
      isUIReadOnly: true,
      targetObjectName: 'workflow',
      targetFieldName: 'runs',
      settings: {
        relationType: RelationType.MANY_TO_ONE,
        onDelete: RelationOnDeleteAction.CASCADE,
        joinColumnName: 'workflowId',
      },
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  favorites: createStandardRelationFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      type: FieldMetadataType.RELATION,
      morphId: null,
      fieldName: 'favorites',
      label: 'Favorites',
      description: 'Favorites linked to the workflow run',
      icon: 'IconHeart',
      isNullable: false,
      isUIReadOnly: true,
      targetObjectName: 'favorite',
      targetFieldName: 'workflowRun',
      settings: {
        relationType: RelationType.ONE_TO_MANY,
      },
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
  timelineActivities: createStandardRelationFieldFlatMetadata({
    objectName,
    workspaceId,
    context: {
      type: FieldMetadataType.RELATION,
      morphId: null,
      fieldName: 'timelineActivities',
      label: 'Timeline Activities',
      description: 'Timeline activities linked to the run',
      icon: 'IconTimelineEvent',
      isNullable: false,
      isUIReadOnly: true,
      targetObjectName: 'timelineActivity',
      targetFieldName: 'targetWorkflowRun',
      settings: {
        relationType: RelationType.ONE_TO_MANY,
      },
    },
    standardObjectMetadataRelatedEntityIds,
    dependencyFlatEntityMaps,
    twentyStandardApplicationId,
    now,
  }),
});
